package ga.core.algorithm.automatic;

import ga.core.GA;
import ga.core.evaluation.IFitnessEvaluator;
import ga.core.goperators.ICrossoverOp;
import ga.core.goperators.IMutationOp;
import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IClusterPopulation;
import ga.core.individual.population.IPopulation;
import ga.core.selection.ISelector;
import ga.core.validation.GAContext;
import ga.core.validation.IValidator;

import java.util.logging.Logger;

/**
 * Steady-State simple genetic algorithm.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class SGA<T extends IIndividual<T>> implements GA<T> {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(SGA.class.getName());

  private static final int CRITICAL_WHILE_ITERATIONS = 1000;

  private final IPopulation<T> population;
  private final ISelector<T> selector;
  private final IMutationOp<T> mutateOperator;
  private final ICrossoverOp<T> crossoverOperator;
  private IValidator<T> validator;
  private final GAContext gaContext;

  private boolean validate;
  private boolean useEliteStrategy;

  private int generation;

  /**
   * Creates a new SGA.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * @param context
   *          The ga context.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SGA(final IPopulation<T> population,
      final IFitnessEvaluator<T> evaluator, final ISelector<T> selector,
      final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final boolean useEliteStrategy,
      final GAContext context) {
    this.population = population;
    this.selector = selector;
    this.mutateOperator = mutateOperator;
    this.crossoverOperator = crossoverOperator;
    this.useEliteStrategy = useEliteStrategy;
    this.gaContext = context;
    population.setEvaluator(evaluator);
  }

  /**
   * Creates a new SGA. The required ga context is newly created.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SGA(final IPopulation<T> population,
      final IFitnessEvaluator<T> evaluator, final ISelector<T> selector,
      final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final boolean useEliteStrategy) {
    this(population, evaluator, selector, mutateOperator, crossoverOperator,
        useEliteStrategy, new GAContext());
  }

  @SuppressWarnings("rawtypes")
  @Override
  public void init() {
    population.initRandomly(validate ? validator : null, gaContext);

    if (population instanceof IClusterPopulation) {
      ((IClusterPopulation) population).doClustering();
    }

    population.evaluateAutomatic();
  }

  @Override
  @SuppressWarnings("rawtypes")
  public void step() {
    generation++;

    int whileCounter = 0;

    IndividualList<T> selectedIndividuals;

    do {
      // SELECT
      selectedIndividuals = selector.select(population);

      // CROSSOVER
      final T ind1 = selectedIndividuals.get(0);
      final T ind2 = selectedIndividuals.get(1);

      selectedIndividuals = crossoverOperator.crossover(ind1, ind2, gaContext);

      // MUTATE
      T mutatedIndividual = mutateOperator.mutate(selectedIndividuals.get(0),
          gaContext);
      selectedIndividuals.set(0, mutatedIndividual);

      mutatedIndividual = mutateOperator.mutate(selectedIndividuals.get(1),
          gaContext);
      selectedIndividuals.set(1, mutatedIndividual);

      whileCounter++;

      if (whileCounter > CRITICAL_WHILE_ITERATIONS) {
        LOGGER.warning("Critical iterations exceeded. Endless loop?");
      }

    } while (validate && validator != null
        && !selectedIndividuals.isValid(validator, gaContext));

    // INSERT
    selector.insert(selectedIndividuals, population, useEliteStrategy);

    if (population instanceof IClusterPopulation) {
      ((IClusterPopulation) population).doClustering();
    }

    // EVALUATE
    population.evaluateAutomatic();
  }

  @Override
  public void setValidate(final boolean validate) {
    if (validate && validator == null) {
      throw new RuntimeException("Error - no validator has been set");
    }

    this.validate = validate;
  }

  @Override
  public void setValidator(final IValidator<T> validator) {
    this.validator = validator;

    if (validator != null) {
      validate = true;
    }
  }

  @Override
  public IPopulation<T> getPopulation() {
    return population;
  }
}
